<?php
// /chat/agent_chat_handler.php

// --------------------------------------------------
// 1) Ativa exibição de erros para debug (remova em produção)
// --------------------------------------------------
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');

// --------------------------------------------------
// 2) Verifica autenticação
// --------------------------------------------------
if (empty($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Não autorizado']);
    exit;
}
$userId = (int) $_SESSION['user_id'];

// --------------------------------------------------
// 3) Lê o JSON de entrada
// --------------------------------------------------
$inputJson = file_get_contents('php://input');
$input     = json_decode($inputJson, true);

if (!is_array($input)) {
    http_response_code(400);
    echo json_encode(['error' => 'Dados inválidos (JSON esperado)']);
    exit;
}

$agentId     = isset($input['agent_id']) ? (int)$input['agent_id'] : 0;
$userMessage = trim($input['message'] ?? '');
$model       = trim($input['model'] ?? 'gpt-3.5-turbo');

if ($agentId <= 0 || $userMessage === '') {
    http_response_code(400);
    echo json_encode(['error' => 'agent_id e message são obrigatórios']);
    exit;
}

// --------------------------------------------------
// 4) Verifica se este Agente pertence a este usuário
// --------------------------------------------------
$stmt = $pdo->prepare('SELECT config_json FROM agents WHERE id = ? AND user_id = ? LIMIT 1');
$stmt->execute([$agentId, $userId]);
$agentRow = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$agentRow) {
    http_response_code(403);
    echo json_encode(['error' => 'Agente não encontrado ou sem permissão']);
    exit;
}

// --------------------------------------------------
// 5) Decodifica o config_json para obter “system_prompt” e “fallback_response”
// --------------------------------------------------
$cfg = json_decode($agentRow['config_json'], true);
if (!is_array($cfg) || empty($cfg['system_prompt'])) {
    http_response_code(500);
    echo json_encode(['error' => 'Configuração do agente inválida']);
    exit;
}

$systemPrompt     = trim($cfg['system_prompt']);
$fallbackResponse = isset($cfg['fallback_response']) ? trim($cfg['fallback_response']) : '';
// (Se houver ações personalizadas, vc pode processar aqui, mas para simplicidade vamos ignorar.)
 
// --------------------------------------------------
// 6) Grava a mensagem do usuário no banco, marcando que pertence a este agente
// --------------------------------------------------
// Para manter histórico separado por agente, criamos OU usamos uma conversa “virtual” associada ao agent_id.
// Para simplificar, vamos gravar em “conversations.agent_id” para sabermos que é conversa de Agente.

try {
    // Primeiro, buscamos se já existe uma conversa “ativa” para este agent_id e user_id.
    $stmtCheck = $pdo->prepare('SELECT id FROM conversations WHERE agent_id = ? AND user_id = ? LIMIT 1');
    $stmtCheck->execute([$agentId, $userId]);
    $convRow = $stmtCheck->fetch(PDO::FETCH_ASSOC);

    if ($convRow) {
        $convId = (int) $convRow['id'];
        // Atualiza updated_at e modelo
        $stmtUpd = $pdo->prepare('UPDATE conversations SET modelo = ?, updated_at = NOW() WHERE id = ?');
        $stmtUpd->execute([$model, $convId]);
    } else {
        // Cria nova conversa vinculada a este agent_id
        $titulo = 'Agent: ' . substr($systemPrompt, 0, 20) . '…';
        $stmtIns = $pdo->prepare('
            INSERT INTO conversations (
              user_id, agent_id, titulo, modelo, created_at, updated_at
            ) VALUES (
              :uid, :aid, :titulo, :modelo, NOW(), NOW()
            )
        ');
        $stmtIns->execute([
            ':uid'     => $userId,
            ':aid'     => $agentId,
            ':titulo'  => $titulo,
            ':modelo'  => $model
        ]);
        $convId = (int) $pdo->lastInsertId();
    }
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha ao criar/atualizar conversa: ' . $e->getMessage()]);
    exit;
}

// Agora gravamos a mensagem do usuário (role = "user") nessa conversa
try {
    $stmtMsg = $pdo->prepare('
        INSERT INTO messages (
          conversation_id, role, content, created_at
        ) VALUES (
          :cid, :role, :conteudo, NOW()
        )
    ');
    $stmtMsg->execute([
        ':cid'      => $convId,
        ':role'     => 'user',
        ':conteudo' => $userMessage
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha ao salvar mensagem do usuário: ' . $e->getMessage()]);
    exit;
}

// --------------------------------------------------
// 7) Monta o contexto para a chamada ao OpenAI
// --------------------------------------------------
// 7.1) Primeiro, adicionamos o “system_prompt” do agente
$messagesForAPI = [
    [
      'role'    => 'system',
      'content' => $systemPrompt
    ]
];

// 7.2) Em seguida, buscamos as últimas N mensagens dessa conversa (incluindo a que acabamos de inserir)
try {
    $stmtHist = $pdo->prepare('
        SELECT role, content
        FROM messages
        WHERE conversation_id = :cid
        ORDER BY created_at DESC
        LIMIT 10
    ');
    $stmtHist->execute([':cid' => $convId]);
    $raw = $stmtHist->fetchAll(PDO::FETCH_ASSOC);
    $historico = array_reverse($raw);
    foreach ($historico as $m) {
        // Pulamos system (já inserido acima); aqui só “user” e “assistant”
        $messagesForAPI[] = [
            'role'    => $m['role'],
            'content' => $m['content']
        ];
    }
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha ao buscar histórico: ' . $e->getMessage()]);
    exit;
}

// --------------------------------------------------
// 8) Chama a API do OpenAI
// --------------------------------------------------
// Aqui vamos usar a **PRÓPRIA CHAVE** do usuário (que está em config.php em OPENAI_API_KEY ou em tabela users).
// Opcionalmente, você já pode ter gravado a chave do usuário em `users.api_key`. Vamos buscá-la:
try {
    $stmtKey = $pdo->prepare('SELECT api_key FROM users WHERE id = ? LIMIT 1');
    $stmtKey->execute([$userId]);
    $rowKey = $stmtKey->fetch(PDO::FETCH_ASSOC);
    if (!$rowKey || empty($rowKey['api_key'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Chave de API do usuário não configurada.']);
        exit;
    }
    $apiKey = trim($rowKey['api_key']);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha ao obter chave de API: ' . $e->getMessage()]);
    exit;
}

$endpoint = 'https://api.openai.com/v1/chat/completions';
$postData = [
    'model'       => $model,
    'messages'    => $messagesForAPI,
    'temperature' => 0.7
];

$ch = curl_init($endpoint);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 60);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json',
    'Authorization: Bearer ' . $apiKey
]);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postData));
$response = curl_exec($ch);

if (curl_errno($ch)) {
    $errMsg = curl_error($ch);
    curl_close($ch);
    http_response_code(500);
    echo json_encode(['error' => 'Erro ao conectar com OpenAI: ' . $errMsg]);
    exit;
}

$httpStatus = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($httpStatus < 200 || $httpStatus >= 300) {
    http_response_code(500);
    echo json_encode(['error' => "OpenAI retornou status HTTP {$httpStatus}"]);
    exit;
}

$apiResult = json_decode($response, true);
if (!isset($apiResult['choices'][0]['message']['content'])) {
    http_response_code(500);
    echo json_encode(['error' => 'Resposta inválida da API do OpenAI']);
    exit;
}

$assistantText = $apiResult['choices'][0]['message']['content'];

// --------------------------------------------------
// 9) Grava a resposta do assistente no banco
// --------------------------------------------------
try {
    $stmtAsst = $pdo->prepare('
        INSERT INTO messages (
          conversation_id, role, content, created_at
        ) VALUES (
          :cid, :role, :conteudo, NOW()
        )
    ');
    $stmtAsst->execute([
        ':cid'      => $convId,
        ':role'     => 'assistant',
        ':conteudo' => $assistantText
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha ao salvar resposta do assistente: ' . $e->getMessage()]);
    exit;
}

// --------------------------------------------------
// 10) Retorna JSON com a resposta do agente
// --------------------------------------------------
echo json_encode([
    'agent_response' => $assistantText,
    'conversation_id'=> $convId,
    'model'          => $model
]);
exit;
