<?php
// /chat/agent_form.php
require_once __DIR__ . '/config.php';
if (empty($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Se existir ID via GET, carrega dados para edição
$agentId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$agentData = [
    'nome'             => '',
    'descricao'        => '',
    'system_prompt'    => '',
    'fallback_response'=> ''
];
$existingActions = []; // será um array de actions vindas do banco
$existingFiles   = []; // será um array de arquivos já enviados

if ($agentId > 0) {
    // 1) Busca dados do agente na tabela agents
    $stmt = $pdo->prepare('SELECT * FROM agents WHERE id = ? AND user_id = ? LIMIT 1');
    $stmt->execute([$agentId, $_SESSION['user_id']]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row) {
        $agentData['nome']             = $row['nome'];
        $agentData['descricao']        = $row['descricao'];
        // Decodifica o JSON que está no banco
        $cfg = json_decode($row['config_json'], true);
        if (isset($cfg['system_prompt'])) {
            $agentData['system_prompt'] = $cfg['system_prompt'];
        }
        if (isset($cfg['fallback_response'])) {
            $agentData['fallback_response'] = $cfg['fallback_response'];
        }
        if (isset($cfg['actions']) && is_array($cfg['actions'])) {
            $existingActions = $cfg['actions'];
        }
    } else {
        echo '<div class="alert alert-error">Agente não encontrado.</div>';
        return;
    }

    // 2) Busca todos os arquivos salvos para este agente
    $stmt2 = $pdo->prepare('
        SELECT id, filename, file_path, uploaded_at
        FROM agent_files
        WHERE agent_id = ?
        ORDER BY uploaded_at DESC
    ');
    $stmt2->execute([$agentId]);
    $existingFiles = $stmt2->fetchAll(PDO::FETCH_ASSOC);
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title><?= $agentId > 0 ? 'Editar Agente' : 'Criar Novo Agente' ?> – Cognis Chat</title>
  <link rel="stylesheet" href="public/assets/css/styles.css" />
</head>
<body data-theme="<?= isset($_COOKIE['theme']) ? $_COOKIE['theme'] : 'light' ?>">
  <?php include __DIR__ . '/views/header.php'; ?>

  <main class="agent-form-wrapper">
    <div class="agent-form-container">
      <div class="form-actions-top" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem;">
        <!-- Botão Cancelar / Voltar -->
        <a href="agent_list.php" class="btn btn-secondary">
          ← Meus Agentes
        </a>
        <!-- Botão de alternar tema claro/escuro -->
        <button id="theme-toggle" class="btn btn-secondary">
          Alternar Tema
        </button>
      </div>

      <h2><?= $agentId > 0 ? 'Editar Agente' : 'Criar Novo Agente' ?></h2>

      <!-- O form envia para o AgentController via POST; o campo config_json será preenchido automaticamente em JS -->
      <form
        id="agent-form"
        action="controllers/AgentController.php"
        method="POST"
        enctype="multipart/form-data"
      >
        <!-- Campo oculto com o ID (se for edição) -->
        <input type="hidden" name="id" value="<?= $agentId ?>" />

        <!-- 1) Nome do Agente -->
        <div class="form-group">
          <label for="agent-name">Nome do Agente:</label>
          <input
            type="text"
            id="agent-name"
            name="nome"
            value="<?= htmlspecialchars($agentData['nome']) ?>"
            required
          />
        </div>

        <!-- 2) Descrição (opcional) -->
        <div class="form-group">
          <label for="agent-desc">Descrição (opcional):</label>
          <textarea
            id="agent-desc"
            name="descricao"
            rows="3"
            placeholder="Para que serve este agente?"
          ><?= htmlspecialchars($agentData['descricao']) ?></textarea>
        </div>

        <!-- 3) Instruções em Prosa -->
        <div class="form-group">
          <label for="system-prompt">Instruções do Agente (prosa):</label>
          <textarea
            id="system-prompt"
            name="system_prompt"
            rows="4"
            placeholder="Explique em linguagem natural como este agente deve se comportar..."
            required
          ><?= htmlspecialchars($agentData['system_prompt']) ?></textarea>
          <small class="field-help">
            Escreva em prosa qual a personalidade, tom de voz e as regras principais do agente.
          </small>
        </div>

        <!-- 4) Fallback Response (opcional) -->
        <div class="form-group">
          <label for="fallback-response">Resposta de Fallback (opcional):</label>
          <textarea
            id="fallback-response"
            name="fallback_response"
            rows="2"
            placeholder="O que o agente deve dizer se não souber responder?"
          ><?= htmlspecialchars($agentData['fallback_response']) ?></textarea>
          <small class="field-help">
            Se quiser que o agente responda algo específico quando estiver fora do escopo, escreva aqui.
          </small>
        </div>

        <!-- 5) Upload de Arquivos de Conhecimento -->
        <div class="form-group">
          <label for="knowledge-files">Arquivos de Conhecimento (txt, md, yaml, csv, pdf etc.):</label>
          <input
            type="file"
            id="knowledge-files"
            name="knowledge_files[]"
            multiple
            accept=".txt,.md,.yaml,.yml,.csv,.pdf"
          />
          <small class="field-help">
            Anexe documentos que servirão como base de conhecimento para este agente.
          </small>

          <!-- 5.1) Se estivermos editando e já houver arquivos, listamos aqui -->
          <?php if (!empty($existingFiles)): ?>
            <div class="existing-files">
              <p>Arquivos já enviados:</p>
              <ul>
                <?php foreach ($existingFiles as $file): ?>
                  <li id="file-item-<?= $file['id'] ?>">
                    <!-- Link para download/visualização -->
                    <a href="<?= htmlspecialchars($file['file_path']) ?>"
                       target="_blank"
                       rel="noopener noreferrer"
                    >
                      <?= htmlspecialchars($file['filename']) ?>
                    </a>
                    <!-- Data de upload -->
                    <span class="upload-date">
                      (Enviado em <?= date('d/m/Y H:i', strtotime($file['uploaded_at'])) ?>)
                    </span>
                    <!-- Botão de remover arquivo -->
                    <button
                      type="button"
                      class="btn btn-danger btn-sm remove-file-btn"
                      data-file-id="<?= $file['id'] ?>"
                    >
                      🗑️
                    </button>
                  </li>
                <?php endforeach; ?>
              </ul>
            </div>
          <?php endif; ?>
        </div>

        <!-- 6) Actions Dinâmicos (sem JSON à vista do usuário) -->
        <div class="form-group">
          <label>Actions Personalizadas:</label>
          <div id="actions-container">
            <!-- Os campos de cada action serão injetados aqui pelo JavaScript -->
          </div>
          <button type="button" id="add-action-btn" class="btn btn-secondary">
            + Adicionar Action
          </button>
          <small class="field-help">
            Cada Action deve ter Nome, Descrição, Endpoint (URL) e Método (GET/POST).
          </small>
        </div>

        <!-- Campo oculto para armazenar o JSON que contêm as actions, system_prompt e fallback_response -->
        <input type="hidden" id="config-json" name="config_json" />

        <!-- 7) Botão para Salvar Agente -->
        <div class="form-group form-group-bottom">
          <button type="submit" id="save-agent-btn" class="btn btn-primary">
            <?= $agentId > 0 ? 'Atualizar Agente' : 'Criar Agente' ?>
          </button>
        </div>
      </form>

      <div id="agent-feedback"></div>
    </div>
  </main>

  <?php include __DIR__ . '/views/footer.php'; ?>

  <!-- JavaScript para controlar Actions, montar config_json, remover arquivos e alternar tema -->
  <script>
    document.addEventListener('DOMContentLoaded', () => {
      // ===== Tema Claro / Escuro =====
      const themeToggleBtn = document.getElementById('theme-toggle');
      const rootEl = document.documentElement;
      function applyTheme(theme) {
        rootEl.setAttribute('data-theme', theme);
        localStorage.setItem('theme', theme);
      }
      // Aplica tema salvo ou detecta inicialmente
      const savedTheme = localStorage.getItem('theme');
      if (savedTheme) {
        applyTheme(savedTheme);
      } else if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
        applyTheme('dark');
      } else {
        applyTheme('light');
      }
      themeToggleBtn.addEventListener('click', () => {
        const current = rootEl.getAttribute('data-theme') === 'dark' ? 'dark' : 'light';
        const next = current === 'dark' ? 'light' : 'dark';
        applyTheme(next);
      });

      // ===== Actions Dinâmicas =====
      const actionsContainer = document.getElementById('actions-container');
      const addActionBtn     = document.getElementById('add-action-btn');
      const configJsonInput  = document.getElementById('config-json');
      const agentForm        = document.getElementById('agent-form');

      let actionsList = [];
      const existingActions = <?= json_encode($existingActions, JSON_UNESCAPED_UNICODE) ?>;
      if (Array.isArray(existingActions) && existingActions.length) {
        actionsList = existingActions.map(act => ({
          name: act.name || '',
          description: act.description || '',
          endpoint: act.endpoint || '',
          method: act.method || 'GET'
        }));
      }

      function renderActions() {
        actionsContainer.innerHTML = '';
        actionsList.forEach((action, index) => {
          const wrapper = document.createElement('div');
          wrapper.classList.add('action-item');
          wrapper.dataset.index = index;

          wrapper.innerHTML = `
            <div class="form-group action-subgroup">
              <label>Nome da Action:</label>
              <input type="text" class="action-name" value="${action.name}" placeholder="Ex: getStockPrice" required />
            </div>
            <div class="form-group action-subgroup">
              <label>Descrição da Action:</label>
              <input type="text" class="action-desc" value="${action.description}" placeholder="Descrição em linguagem natural" required />
            </div>
            <div class="form-group action-subgroup">
              <label>Endpoint (URL):</label>
              <input type="url" class="action-endpoint" value="${action.endpoint}" placeholder="https://api.exemplo.com/recurso" required />
            </div>
            <div class="form-group action-subgroup">
              <label>Método HTTP:</label>
              <select class="action-method">
                <option value="GET" ${action.method === 'GET' ? 'selected' : ''}>GET</option>
                <option value="POST" ${action.method === 'POST' ? 'selected' : ''}>POST</option>
              </select>
            </div>
            <button type="button" class="btn btn-danger remove-action-btn">
              ❌ Remover Action
            </button>
            <hr />
          `;

          wrapper.querySelector('.remove-action-btn').addEventListener('click', () => {
            actionsList.splice(index, 1);
            renderActions();
          });

          wrapper.querySelector('.action-name').addEventListener('input', e => {
            actionsList[index].name = e.target.value;
          });
          wrapper.querySelector('.action-desc').addEventListener('input', e => {
            actionsList[index].description = e.target.value;
          });
          wrapper.querySelector('.action-endpoint').addEventListener('input', e => {
            actionsList[index].endpoint = e.target.value;
          });
          wrapper.querySelector('.action-method').addEventListener('change', e => {
            actionsList[index].method = e.target.value;
          });

          actionsContainer.appendChild(wrapper);
        });
      }

      addActionBtn.addEventListener('click', () => {
        actionsList.push({
          name: '',
          description: '',
          endpoint: '',
          method: 'GET'
        });
        renderActions();
      });

      agentForm.addEventListener('submit', e => {
        for (let i = 0; i < actionsList.length; i++) {
          const a = actionsList[i];
          if (!a.name.trim() || !a.description.trim() || !a.endpoint.trim()) {
            alert('Preencha todos os campos de cada Action antes de enviar.');
            e.preventDefault();
            return;
          }
        }

        const systemPrompt = document.getElementById('system-prompt').value.trim();
        const fallbackResp = document.getElementById('fallback-response').value.trim();

        const cfg = { system_prompt: systemPrompt };
        if (fallbackResp !== '') {
          cfg.fallback_response = fallbackResp;
        }
        if (actionsList.length > 0) {
          cfg.actions = actionsList;
        }

        configJsonInput.value = JSON.stringify(cfg, null, 2);
      });

      renderActions();

      // ===== Remover Arquivos Existentes via AJAX =====
      document.querySelectorAll('.remove-file-btn').forEach(btn => {
        btn.addEventListener('click', () => {
          const fileId = btn.getAttribute('data-file-id');
          if (!fileId) return;
          if (!confirm('Tem certeza que deseja remover este arquivo?')) return;

          fetch('controllers/DeleteAgentFile.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ file_id: fileId })
          })
          .then(res => res.json())
          .then(data => {
            if (data.success) {
              const li = document.getElementById('file-item-' + fileId);
              if (li) li.remove();
            } else {
              alert('Falha ao remover o arquivo: ' + (data.error || ''));
            }
          })
          .catch(err => {
            console.error(err);
            alert('Erro ao remover o arquivo.');
          });
        });
      });
    });
  </script>
</body>
</html>
