<?php
// /chat/chat_handler.php

// --------------------------------------------------
// 1) Ativa exibição de erros para debug (remova em produção)
// --------------------------------------------------
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');

// --------------------------------------------------
// 2) Verifica autenticação
// --------------------------------------------------
if (empty($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Não autorizado']);
    exit;
}

$userId = (int) $_SESSION['user_id'];

// --------------------------------------------------
// 3) Busca a chave de API associada ao usuário
// --------------------------------------------------
try {
    $stmt = $pdo->prepare('SELECT api_key FROM users WHERE id = :uid LIMIT 1');
    $stmt->execute([':uid' => $userId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row || empty($row['api_key'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Chave da API não configurada. Vá em Configurações da Conta e adicione sua chave.']);
        exit;
    }
    $apiKey = trim($row['api_key']);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha ao buscar chave de API: ' . $e->getMessage()]);
    exit;
}

// --------------------------------------------------
// 4) Lê o JSON de entrada
// --------------------------------------------------
$inputJson = file_get_contents('php://input');
$input     = json_decode($inputJson, true);

// Verifica se veio JSON válido
if (!is_array($input)) {
    http_response_code(400);
    echo json_encode(['error' => 'Dados inválidos (JSON esperado)']);
    exit;
}

$convId      = isset($input['conversation_id']) ? (int) $input['conversation_id'] : 0;
$userMessage = trim($input['message'] ?? '');
$model       = trim($input['model'] ?? 'gpt-3.5-turbo');

// Captura possível agent_id para injetar system_prompt
$agentId = isset($input['agent_id']) ? (int) $input['agent_id'] : 0;

// Mensagem vazia não é permitida
if ($userMessage === '') {
    http_response_code(400);
    echo json_encode(['error' => 'Mensagem vazia']);
    exit;
}

// --------------------------------------------------
// 5) Se conversation_id = 0, cria nova conversa
// --------------------------------------------------
if ($convId <= 0) {
    try {
        $tituloPadrao = 'Nova Conversa';
        $stmt = $pdo->prepare('
            INSERT INTO conversations (
                user_id,
                titulo,
                modelo,
                created_at,
                updated_at
            ) VALUES (
                :uid, :titulo, :modelo, NOW(), NOW()
            )
        ');
        $stmt->execute([
            ':uid'     => $userId,
            ':titulo'  => $tituloPadrao,
            ':modelo'  => $model
        ]);
        $convId = (int) $pdo->lastInsertId();
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Falha ao criar conversa: ' . $e->getMessage()
        ]);
        exit;
    }
} else {
    // --------------------------------------------------
    // 6) Se já existe conversa, atualiza o campo "modelo"
    // --------------------------------------------------
    try {
        $stmt = $pdo->prepare('
            UPDATE conversations
            SET modelo = :modelo, updated_at = NOW()
            WHERE id = :cid AND user_id = :uid
        ');
        $stmt->execute([
            ':modelo' => $model,
            ':cid'    => $convId,
            ':uid'    => $userId
        ]);
        if ($stmt->rowCount() === 0) {
            http_response_code(403);
            echo json_encode(['error' => 'Conversa não encontrada ou sem permissão']);
            exit;
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Falha ao atualizar modelo: ' . $e->getMessage()
        ]);
        exit;
    }
}

// --------------------------------------------------
// 7) Grava a mensagem do usuário em `messages`
// --------------------------------------------------
try {
    $stmt = $pdo->prepare('
        INSERT INTO messages (
            conversation_id,
            role,
            content,
            created_at
        ) VALUES (
            :cid, :role, :conteudo, NOW()
        )
    ');
    $stmt->execute([
        ':cid'      => $convId,
        ':role'     => 'user',
        ':conteudo' => $userMessage
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Falha ao salvar mensagem do usuário: ' . $e->getMessage()
    ]);
    exit;
}

// --------------------------------------------------
// 8) Lê as últimas 10 mensagens desta conversa
// --------------------------------------------------
try {
    $stmt = $pdo->prepare('
        SELECT role, content
        FROM messages
        WHERE conversation_id = :cid
        ORDER BY created_at DESC
        LIMIT 10
    ');
    $stmt->execute([':cid' => $convId]);
    $ultimasRaw = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $ultimas    = array_reverse($ultimasRaw, true);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Falha ao buscar histórico de mensagens: ' . $e->getMessage()
    ]);
    exit;
}

$messagesForAPI = [];

// --------------------------------------------------
// 9) Se veio agent_id, busca o system_prompt e injetável
// --------------------------------------------------
if ($agentId > 0) {
    try {
        $stmtA = $pdo->prepare('SELECT config_json FROM agents WHERE id = ? AND user_id = ? LIMIT 1');
        $stmtA->execute([$agentId, $userId]);
        $rowA = $stmtA->fetch(PDO::FETCH_ASSOC);
        if ($rowA) {
            $cfg = json_decode($rowA['config_json'], true);
            if (isset($cfg['system_prompt']) && trim($cfg['system_prompt']) !== '') {
                $messagesForAPI[] = [
                    'role'    => 'system',
                    'content' => $cfg['system_prompt']
                ];
            }
        }
    } catch (PDOException $e) {
        // Se não conseguir buscar config do agente, seguimos sem system_prompt
    }
}

// --------------------------------------------------
// 10) Adiciona o histórico (user + assistant) ao array
// --------------------------------------------------
foreach ($ultimas as $m) {
    $messagesForAPI[] = [
        'role'    => $m['role'],
        'content' => $m['content']
    ];
}
// (Observação: a mensagem do usuário já está no histórico recuperado de "ultimas")
// Portanto, não há necessidade de re-adicionar $userMessage separadamente aqui.

// --------------------------------------------------
// 11) Chama a API do OpenAI usando a chave do usuário
// --------------------------------------------------
$endpoint = 'https://api.openai.com/v1/chat/completions';
$postData = [
    'model'       => $model,
    'messages'    => $messagesForAPI,
    'temperature' => 0.7
    // Se você quiser incluir actions definidas no config_json, adicione um campo "functions" aqui
];

$ch = curl_init($endpoint);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 60);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json',
    'Authorization: Bearer ' . $apiKey
]);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postData));
$response = curl_exec($ch);

if (curl_errno($ch)) {
    $errMsg = curl_error($ch);
    curl_close($ch);
    http_response_code(500);
    echo json_encode(['error' => 'Erro ao conectar com OpenAI: ' . $errMsg]);
    exit;
}

$httpStatus = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($httpStatus < 200 || $httpStatus >= 300) {
    http_response_code(500);
    echo json_encode(['error' => "OpenAI retornou status HTTP {$httpStatus}"]);
    exit;
}

$apiResult = json_decode($response, true);
if (!isset($apiResult['choices'][0]['message']['content'])) {
    http_response_code(500);
    echo json_encode(['error' => 'Resposta inválida da API do OpenAI']);
    exit;
}

$assistantText = $apiResult['choices'][0]['message']['content'];

// --------------------------------------------------
// 12) Grava a resposta do assistente no banco
// --------------------------------------------------
try {
    $stmt = $pdo->prepare('
        INSERT INTO messages (
            conversation_id,
            role,
            content,
            created_at
        ) VALUES (
            :cid, :role, :conteudo, NOW()
        )
    ');
    $stmt->execute([
        ':cid'      => $convId,
        ':role'     => 'assistant',
        ':conteudo' => $assistantText
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Falha ao salvar resposta do assistente: ' . $e->getMessage()
    ]);
    exit;
}

// --------------------------------------------------
// 13) Retorna JSON com a resposta e o ID da conversa
// --------------------------------------------------
echo json_encode([
    'assistant_response' => $assistantText,
    'conversation_id'    => $convId,
    'model'              => $model
]);
exit;
?>
