document.addEventListener('DOMContentLoaded', () => {
  // 1) Referências a elementos
  const toggleDarkmode = document.getElementById('toggle-darkmode');
  const body = document.body;

  const btnNewConversation = document.getElementById('btn-new-conversation');
  const btnNewAgent = document.getElementById('btn-new-agent');
  const conversationList = document.getElementById('conversation-list');
  const agentList = document.getElementById('agent-list');

  const messagesContainer = document.getElementById('messages');
  const inputMessage = document.getElementById('input-message');
  const sendBtn = document.getElementById('send-btn');
  const modelSelector = document.getElementById('model-selector');

  const agentModal = document.getElementById('agent-modal');
  const agentForm = document.getElementById('agent-form');
  const agentCancel = document.getElementById('agent-cancel');

  // 2) Estado em memória (pode ser localStorage ou variáveis JS)
  let currentConversationId = null; // ID de conversa corrente
  let conversations = {};          // { id: [ {role:'user'|'assistant', content: '...'} ] }
  let agents = {};                // { id: { nome, modelo, temperatura, instrucoes } }
  let selectedAgentId = null;

  // 3) Carregar agentes salvos de data/agents.json via fetch AJAX
  fetch('api/agents.php')
    .then(res => res.json())
    .then(data => {
      agents = data; // formato: objeto cujas chaves são IDs
      renderAgentList();
    });

  // 4) Função para renderizar lista de agentes na sidebar
  function renderAgentList() {
    agentList.innerHTML = ''; 
    for (const id in agents) {
      const ag = agents[id];
      const li = document.createElement('li');
      li.textContent = ag.nome;
      li.dataset.agentId = id;
      li.addEventListener('click', () => {
        activateAgent(id);
      });
      // Adicionar botão de editar e deletar (ícones SVG ou texto)
      const btnEdit = document.createElement('button');
      btnEdit.textContent = '✏️';
      btnEdit.classList.add('btn-secondary');
      btnEdit.addEventListener('click', (e) => {
        e.stopPropagation();
        openEditAgentModal(id);
      });
      const btnDelete = document.createElement('button');
      btnDelete.textContent = '❌';
      btnDelete.classList.add('btn-secondary');
      btnDelete.addEventListener('click', (e) => {
        e.stopPropagation();
        deleteAgent(id);
      });
      li.append(btnEdit, btnDelete);
      agentList.appendChild(li);
    }
  }

  // 5) Alternar modo claro/escuro
  toggleDarkmode.addEventListener('change', () => {
    if (toggleDarkmode.checked) {
      body.classList.add('dark-mode');
      body.classList.remove('light-mode');
      localStorage.setItem('theme', 'dark');
    } else {
      body.classList.add('light-mode');
      body.classList.remove('dark-mode');
      localStorage.setItem('theme', 'light');
    }
  });

  // No carregamento, aplica tema salvo
  const savedTheme = localStorage.getItem('theme');
  if (savedTheme === 'dark') {
    toggleDarkmode.checked = true;
    body.classList.add('dark-mode');
    body.classList.remove('light-mode');
  } else {
    toggleDarkmode.checked = false;
    body.classList.add('light-mode');
    body.classList.remove('dark-mode');
  }

  // 6) Criar nova conversa
  btnNewConversation.addEventListener('click', () => {
    currentConversationId = `conv_${Date.now()}`;
    conversations[currentConversationId] = [];
    renderConversationList();
    loadConversation(currentConversationId);
  });

  function renderConversationList() {
    conversationList.innerHTML = '';
    for (const convId in conversations) {
      const li = document.createElement('li');
      li.textContent = convId; // você pode usar título customizado depois
      li.dataset.convId = convId;
      li.addEventListener('click', () => {
        loadConversation(convId);
      });
      conversationList.appendChild(li);
    }
  }

  function loadConversation(convId) {
    currentConversationId = convId;
    messagesContainer.innerHTML = '';
    const msgs = conversations[convId] || [];
    msgs.forEach(msg => appendMessage(msg.role, msg.content));
  }

  function appendMessage(role, content) {
    const div = document.createElement('div');
    div.classList.add('message', role === 'user' ? 'user' : 'ai');
    div.innerHTML = content; // se quiser converter markdown em HTML, pode usar alguma lib leve em JS
    messagesContainer.appendChild(div);
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
  }

  // 7) Enviar mensagem (click ou Enter)
  sendBtn.addEventListener('click', sendMessage);
  inputMessage.addEventListener('keydown', (e) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  });

  function sendMessage() {
    const text = inputMessage.value.trim();
    if (!text || !currentConversationId) return;

    // 7.1) Adiciona mensagem do usuário ao histórico e à UI
    conversations[currentConversationId].push({ role: 'user', content: text });
    appendMessage('user', text);
    inputMessage.value = '';
    inputMessage.rows = 1;

    // 7.2) Prepara payload para o backend (chat.php)
    const selectedModel = modelSelector.value;
    let systemPrompt = null;
    if (selectedAgentId && agents[selectedAgentId]) {
      systemPrompt = agents[selectedAgentId].instrucoes;
    }

    const payload = {
      model: selectedModel,
      messages: []
    };
    if (systemPrompt) {
      payload.messages.push({ role: 'system', content: systemPrompt });
    }
    conversations[currentConversationId].forEach(msg => {
      payload.messages.push({ role: msg.role, content: msg.content });
    });

    // 7.3) Faz chamada AJAX para chat.php
    fetch('api/chat.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload)
    })
    .then(res => res.json())
    .then(data => {
      const aiContent = data.choices[0].message.content;
      conversations[currentConversationId].push({ role: 'assistant', content: aiContent });
      appendMessage('ai', aiContent);
    })
    .catch(err => {
      console.error(err);
      appendMessage('ai', '<em>Erro ao obter resposta da API.</em>');
    });
  }

  // 8) Ajustar altura do textarea automaticamente
  inputMessage.addEventListener('input', () => {
    inputMessage.style.height = 'auto';
    inputMessage.style.height = (inputMessage.scrollHeight) + 'px';
  });

  // 9) Agent Builder: abrir modal para criar novo agente
  btnNewAgent.addEventListener('click', () => {
    selectedAgentId = null;
    openAgentModal();
  });

  function openAgentModal(agentId = null) {
    if (agentId && agents[agentId]) {
      const ag = agents[agentId];
      document.getElementById('agent-id').value = agentId;
      document.getElementById('agent-name').value = ag.nome;
      document.getElementById('agent-model').value = ag.modelo;
      document.getElementById('agent-temp').value = ag.temperatura;
      document.getElementById('agent-instructions').value = ag.instrucoes;
    } else {
      document.getElementById('agent-id').value = '';
      document.getElementById('agent-form').reset();
    }
    agentModal.classList.remove('hidden');
  }

  agentCancel.addEventListener('click', () => {
    agentModal.classList.add('hidden');
  });

  // 10) Submeter formulário de agente (criar ou editar)
  agentForm.addEventListener('submit', (e) => {
    e.preventDefault();
    const idField = document.getElementById('agent-id').value;
    const nome = document.getElementById('agent-name').value.trim();
    const modelo = document.getElementById('agent-model').value;
    const temperatura = parseFloat(document.getElementById('agent-temp').value);
    const instrucoes = document.getElementById('agent-instructions').value.trim();

    const payload = { nome, modelo, temperatura, instrucoes };
    let url = 'api/agents.php';
    let method = 'POST';

    if (idField) {
      payload.id = idField;
      method = 'PUT';
    }

    fetch(url, {
      method,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload)
    })
    .then(res => res.json())
    .then(data => {
      agents = data;
      renderAgentList();
      agentModal.classList.add('hidden');
    })
    .catch(err => console.error(err));
  });

  // 11) Função para ativar um agente
  function activateAgent(agentId) {
    selectedAgentId = agentId;
  }

  // 12) Excluir agente
  function deleteAgent(agentId) {
    fetch('api/agents.php', {
      method: 'DELETE',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ id: agentId })
    })
    .then(res => res.json())
    .then(data => {
      agents = data;
      renderAgentList();
      if (selectedAgentId === agentId) {
        selectedAgentId = null;
      }
    })
    .catch(err => console.error(err));
  }
});
